/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F4011"  Board */
/* Hardware    : ET-BASE dsPIC30F4011                 */
/* Target MCU  : dsPIC30F4011                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Use ADC0                     */
/*             : Read ADC0 & Display to UART1         */
/******************************************************/
/* Used RB0 = ADC Input                               */
/* Display Result to UART1 : 9600,N,8,1               */
/******************************************************/

#include <p30f4011.h>                                		// dsPIC30F4011 MPU Register
#include <stdio.h>											// Used "sprintf" Function
#include "adc10.h"											// Used ADC Library Function
#include "uart.h"											// Used UART Config

/* Setup Configuration For ET-BASE dsPIC30F4011 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    // Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End   Configuration For ET-BASE dsPIC30F4011 */

char  uart_buf[40];											// "sprintf" Buffer
unsigned int  adc_buff;										// ADC Result
unsigned int  adc_count;

/* pototype  section */
void init_uart(void);										// Initial UART1 Function
void init_adc(void);										// Initial ADC Function

int main(void)
{  
  init_uart();												// Initial UART = 9600,N,8,1
  init_adc();												// Initial ADC[0..3]
  	
  sprintf(uart_buf,"\n\n\rET-BASE dsPIC30F4011 : ADC\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo ADC[0..3] Scan Read Input\n\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  

  // Strat Read ADC Continue //
  while(1)													// Loop Continue
  {
    ADCON1bits.SAMP = 1;				    				// Start Sampling ADC Now
    while(!ADCON1bits.SAMP);								// Wait ADC Sampling Complete
    ConvertADC10();											// Convert Result
 	while(BusyADC10());										// Wait Conver Result Complete    

    adc_buff = ReadADC10(0);								// Save Result to Buffer    

    // Display ADC[0] Result on UART1
    sprintf(uart_buf,"\rADC[0] = "); 						// Print Message String 
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete  

    // Display Result ADC0    	
    sprintf(uart_buf,"%4d : \r",adc_buff);					// 4 Digit Decimal[0..1023] Display
    putsUART1((unsigned int *)uart_buf);                    // Print uart_buff to UART1
    while(BusyUART1());										// Wait putsUART1 Complete 
  } 
}	  

/*********************************/
/* Initial UART for dsPIC30F4011 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart()
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  
  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
  			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-BASE dsPIC30F4011 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-BASE dsPIC30F4011 UART Baudrate = 9600 BPS
}

/********************************/
/* Initial ADC for dsPIC30F4011 */
/* Used RB0 = 10 Bit ADC (ADC0) */
/********************************/
void init_adc()
{
  ADCON1bits.ADON = 0;										// Turn-OFF ADC Before Change Config

  // Initial ADC Channel	
  SetChanADC10(ADC_CH0_POS_SAMPLEA_AN0 &					// ADC0 = Input(+)			   
			   ADC_CH0_NEG_SAMPLEA_NVREF);					// GND  = Input(-)             

  // Initial ADC Interrupt
  ConfigIntADC10(ADC_INT_DISABLE);							// Dsiable ADC Interrupt
  
  // Open ADC Function & Turn ON ADC
  OpenADC10(ADC_MODULE_ON &									// Turn-ON ADC Function
       		ADC_IDLE_STOP &									// Stop ADC in IDLE Mode
			ADC_FORMAT_INTG &								// Result Format = Unsigned Integer 
			ADC_CLK_MANUAL &								// ADC Clock = Manual
			ADC_AUTO_SAMPLING_ON &							// Enable ADC Sampling 
            ADC_SAMPLE_SIMULTANEOUS,						// Sample Style = Simultaneous			

            // ADC Config2
			ADC_VREF_AVDD_AVSS &							// VDD=Vref(H),VSS=Vref(L)
			ADC_SCAN_ON &									// Enable Scan        
            ADC_CONVERT_CH0 &								// Used ADC0 to Convert Result
            ADC_SAMPLES_PER_INT_1 &		    				// Number of Sample Between Interrupt
            ADC_ALT_BUF_OFF & 								// Disable Alternate Buffer
			ADC_ALT_INPUT_OFF ,								// Disable Alternate Input	          

			// ADC Config3
			ADC_SAMPLE_TIME_1 &	    						// Samplink Time = Fast
			ADC_CONV_CLK_INTERNAL_RC &						// Used Internal RC Clock Sampling			
			ADC_CONV_CLK_Tcy,								// Conversion Clock = Fast

			// ADC Config Port = Enable RB0 = ADC0
			ENABLE_AN0_ANA , 								// Enable RB0 = ADC0
		            
			// ADC Config Scan = ON ADC0 ,OFF ADC[1..7]
            SKIP_SCAN_AN1 & 								// Disable Scan ADC1
			SKIP_SCAN_AN2 & 								// Disable Scan ADC2
			SKIP_SCAN_AN3 & 								// Disable Scan ADC3
			SKIP_SCAN_AN4 & 								// Disable Scan ADC4
		    SKIP_SCAN_AN5 &									// Disable Scan ADC5
			SKIP_SCAN_AN6 & 								// Disable Scan ADC6
			SKIP_SCAN_AN7);									// Disable Scan ADC7
}
